function [F,Fd,P,E,Ed,U,fptot,mincorr] = DFM1(X,r,output)

% Fits the Data Factor Model I: Minimize ssq(X-F*P'-E*U) with constraints 
% [F E]'*[F E]/n=eye(k+r), ones(1,n)*[F E]=zeros(1,k+r), U diagonal.
% Data X should be centered.
%
% Stegeman, A. (2016). A new method for simultaneous estimation of the 
% factor model parameters, factor scores, and unique parts.
%
% Output:           Fd          determinate part of factors F
%                   Ed          determinate part of unique part E
%                   U           diagonal matrix of unique stds (cols of E have variance 1)
%                   fptot       100-100*ssq(X-F*P'-E*U)/ssq(X)
%                   mincorr     minimal correlation between two alternative
%                               factors (measure of factor indeterminacy)
%
% Note: multiple random starts may be needed to overcome local optima !

[n k]=size(X);
conv=1e-6;

% initialize 
F=randn(n,r);
F=F-ones(n,n)*F/n;
E=randn(n,k);
E=E-ones(n,n)*E/n;

P=(X-E)'*F*inv(F'*F);
U=zeros(k,k);
XFP=X-F*P';
for j=1:k
    U(j,j)=inv(E(:,j)'*E(:,j))*E(:,j)'*XFP(:,j);
end


f=ssq(X-F*P'-E*U);
fprintf(' Function value at Start is %12.8f \n',f);

fold=f+2*conv*f;
iter=0;


while fold-f>conv*f

% update F and E
           
[Q1,D,Q2]=svd(X*[P U],0);
Q1d=Q1(:,1:k);                  % determinate parts
Q2d=Q2(:,1:k);
space1=orth(null([Q1d ones(n,1)]'));
space2=orth(null(Q2d'));
Q1(:,k+1:k+r)=orth(space1*randn(size(space1,2),r));   % set random indeterminate part of Q1 to mean zero
Q2(:,k+1:k+r)=orth(space2*randn(size(space2,2),r));   % random indeterminate part of Q2

G=Q1*Q2';                     % determinate + indeterminate parts of F and E
F=sqrt(n)*G(:,1:r);
E=sqrt(n)*G(:,r+1:k+r);


% update P
P=X'*F/n;
    
% update U
U=diag(diag(E'*X))/n;


% evaluate
fold=f;
f=ssq(X-F*P'-E*U);
iter=iter+1;
if rem(iter,1)==0
    fprintf('f= %12.8f after %g iters ; diff.=%12.8f \n',f,iter,(fold-f));
end;

           
end

%fprintf(' Function value is %12.8f after %g iterations \n\n',f,iter);
fptot=100-100*f/ssq(X);

% correct negative values of U
for j=1:k
    if U(j,j)<0
        U(j,j)=-U(j,j);
        E(:,j)=-E(:,j);
    end
end

Fd=X*inv(X'*X)*X'*F;
Ed=X*inv(X'*X)*X'*E;

if output==1
    fp=100-100*ssq(X-F*P'-E*U)/ssq(X);
    fprintf(' Fit of F*P''+E*U to X equals %5.2f percent \n\n',fp);
    fp=100-100*ssq(X-F*P')/ssq(X);
    fprintf(' Fit of F*P'' to X equals %5.2f percent \n\n',fp);
    fp=100-100*ssq(X-E*U)/ssq(X);
    fprintf(' Fit of E*U to X equals %5.2f percent \n\n',fp);
    fp=100-100*ssq(X-F*P'-E*U)/ssq(X-F*P');
    fprintf(' Fit of E*U to X-F*P'' equals %5.2f percent \n\n',fp);
    fp=100-100*ssq(X-E*U-F*P')/ssq(X-E*U);
    fprintf(' Fit of F*P'' to X-E*U equals %5.2f percent \n\n',fp);
end


% sign changes in P and F such that largest entry per column in P is positive
temp=sort(P./(ones(k,1)*max(abs(P))));
signs=ones(1,r)-2*(temp(1,:)==-1);
P=P*diag(signs);      
F=F*diag(signs);
Fd=Fd*diag(signs);


% compute mincorr
mincorr=2*diag(Fd'*Fd)/n-1;

               
if output==1
    % check postive semi-definiteness of X'*X/n-U*E'E*U/n  
    fprintf(' Eigenvalues of X''*X/n-U*U are: ');
    e=eig(X'*X/n-U*U);
    e'
end

end

