function [F,Fd,P,E,Ed,U,comm,ecv,ecv_total,fptot,mincorr] = DFM2(X,r,output)

% Fits the Data Factor Model II: Minimize ssq(X-F*P'-E*U) with constraints 
% [F E]'*[F E]/n=eye(k+r), ones(1,n)*[F E]=zeros(1,k+r), U diagonal, and
% E'*(X-E*U)=zeros(k,k). Data X should be centered.
%
% Stegeman, A. (2016). A new method for simultaneous estimation of the 
% factor model parameters, factor scores, and unique parts.
%
% Output:           Fd          determinate part of factors F
%                   Ed          determinate part of unique part E
%                   U           diagonal matrix of unique stds (cols of E have variance 1)
%                   comm        vector of communalities (diagonal of X'*X/n-U*U)
%                   ecv         vector of explained common variances per variable (percentages)
%                   ecv_total   total explained common variance (percentage)
%                   fptot       100-100*ssq(X-F*P'-E*U)/ssq(X)
%                   mincorr     minimal correlation between two alternative
%                               factors (measure of factor indeterminacy)


[n k]=size(X);
conv=1e-6;
S=X'*X/n;

% estimate U and compute Ed
[comm,errvar,f]=mrfa(S,r,1e-4,1e-4);
U=real(sqrt(diag(errvar)));
Ed=X*inv(S)*U;

% compute Eu and E
[Q1,S1,Q2]=svd(n*eye(k)-Ed'*Ed,0);
r1=rank(n*eye(k)-Ed'*Ed);
%r1=sum(diag(S1)>0.001);
G=orth(null([X ones(n,1)]')); 
M=orth(randn(size(G,2),r1));
Eu=G*M*S1(1:r1,1:r1)^(0.5)*Q1(:,1:r1)';
E=Ed+Eu;

% estimate F and P (by truncated SVD of X-E*U)
[V1,D1,V2]=svd(X-E*U,0);
F=sqrt(n)*V1(:,1:r);
P=V2(:,1:r)*D1(1:r,1:r)/sqrt(n);
Fd=X*inv(X'*X)*X'*F;


% evaluate
fprintf('\n');
fprintf(' SSQ function value is %12.8f \n\n',f);
fptot=100-100*f/ssq(X);

if output==1
    fp=100-100*ssq(X-F*P'-E*U)/ssq(X);
    fprintf(' Fit of F*P''+E*U to X equals %5.2f percent \n\n',fp);
    fp=100-100*ssq(X-F*P')/ssq(X);
    fprintf(' Fit of F*P'' to X equals %5.2f percent \n\n',fp);
    fp=100-100*ssq(X-E*U)/ssq(X);
    fprintf(' Fit of E*U to X equals %5.2f percent \n\n',fp);
    fp=100-100*ssq(X-F*P'-E*U)/ssq(X-F*P');
    fprintf(' Fit of E*U to X-F*P'' equals %5.2f percent \n\n',fp);
    fp=100-100*ssq(X-E*U-F*P')/ssq(X-E*U);
    fprintf(' Fit of F*P'' to X-E*U equals %5.2f percent \n\n',fp);
end


% sign changes in P and F such that largest entry per column in P is positive
temp=sort(P./(ones(k,1)*max(abs(P))));
signs=ones(1,r)-2*(temp(1,:)==-1);
P=P*diag(signs);      
F=F*diag(signs);
Fd=Fd*diag(signs);


% compute mincorr
mincorr=2*diag(Fd'*Fd)/n-1;

               
% compute explained common variances
comm=diag((X-E*U)'*(X-E*U)/n);          
ecv=100*(diag(P*P'))./comm;        
ecv_total=100*trace(P*P')/sum(comm);

if output==1
    fprintf(' Total explained common variance is %5.2f percent \n\n',ecv_total);
    % check postive semi-definiteness of X'*X/n-U*E'E*U/n
    fprintf(' Eigenvalues of X''*X/n-U*U are: ');
    e=eig(X'*X/n-U*U);
    e'
end

end